#include "hbapi.h"
#include "hbpcode.h"

#define OPTYPE_CODE      1
#define OPTYPE_PARAM     2
#define OPTYPE_JUMP      4
#define OPTYPE_JUMPIN    8

typedef struct
{
   BYTE*     pPCode;
   ULONG     ulPCodeLen;
   BYTE*     pOpType;
   ULONG     ulLocals;
   ULONG     ulParams;
   char*     szFunc;
} PCODECONTEXT;


const BYTE s_bPCodeLen[] = {
   1,   /* HB_P_AND,                  */
   1,   /* HB_P_ARRAYPUSH,            */
   1,   /* HB_P_ARRAYPOP,             */
   3,   /* HB_P_ARRAYDIM,             */
   3,   /* HB_P_ARRAYGEN,             */
   1,   /* HB_P_EQUAL,                */
   1,   /* HB_P_ENDBLOCK,             */
   1,   /* HB_P_ENDPROC,              */
   1,   /* HB_P_EXACTLYEQUAL,         */
   1,   /* HB_P_FALSE,                */
   1,   /* HB_P_FORTEST,              */
   3,   /* HB_P_FUNCTION,             */
   2,   /* HB_P_FUNCTIONSHORT,        */
   3,   /* HB_P_FRAME,                */
   1,   /* HB_P_FUNCPTR,              */
   1,   /* HB_P_GREATER,              */
   1,   /* HB_P_GREATEREQUAL,         */
   1,   /* HB_P_DEC,                  */
   1,   /* HB_P_DIVIDE,               */
   3,   /* HB_P_DO,                   */
   2,   /* HB_P_DOSHORT,              */
   1,   /* HB_P_DUPLICATE,            */
   1,   /* HB_P_DUPLTWO,              */
   1,   /* HB_P_INC,                  */
   1,   /* HB_P_INSTRING,             */
   2,   /* HB_P_JUMPNEAR,             */
   3,   /* HB_P_JUMP,                 */
   4,   /* HB_P_JUMPFAR,              */
   2,   /* HB_P_JUMPFALSENEAR,        */
   3,   /* HB_P_JUMPFALSE,            */
   4,   /* HB_P_JUMPFALSEFAR,         */
   2,   /* HB_P_JUMPTRUENEAR,         */
   3,   /* HB_P_JUMPTRUE,             */
   4,   /* HB_P_JUMPTRUEFAR,          */
   1,   /* HB_P_LESSEQUAL,            */
   1,   /* HB_P_LESS,                 */
   3,   /* HB_P_LINE,                 */
   0,   /* HB_P_LOCALNAME,            */
   2,   /* HB_P_MACROPOP,             */
   2,   /* HB_P_MACROPOPALIASED,      */
   2,   /* HB_P_MACROPUSH,            */
   3,   /* HB_P_MACROARRAYGEN,        */
   2,   /* HB_P_MACROPUSHLIST,        */
   1,   /* HB_P_MACROPUSHINDEX,       */
   2,   /* HB_P_MACROPUSHPARE,        */
   2,   /* HB_P_MACROPUSHALIASED,     */
   1,   /* HB_P_MACROSYMBOL,          */
   1,   /* HB_P_MACROTEXT,            */
   3,   /* HB_P_MESSAGE,              */
   1,   /* HB_P_MINUS,                */
   1,   /* HB_P_MODULUS,              */
   0,   /* HB_P_MODULENAME,           */
   3,   /* HB_P_MMESSAGE,             */
   3,   /* HB_P_MPOPALIASEDFIELD,     */
   3,   /* HB_P_MPOPALIASEDVAR,       */
   3,   /* HB_P_MPOPFIELD,            */
   3,   /* HB_P_MPOPMEMVAR,           */
   3,   /* HB_P_MPUSHALIASEDFIELD,    */
   3,   /* HB_P_MPUSHALIASEDVAR,      */
   0,   /* HB_P_MPUSHBLOCK,           */
   3,   /* HB_P_MPUSHFIELD,           */
   3,   /* HB_P_MPUSHMEMVAR,          */
   3,   /* HB_P_MPUSHMEMVARREF,       */
   3,   /* HB_P_MPUSHSYM,             */
   3,   /* HB_P_MPUSHVARIABLE,        */
   1,   /* HB_P_MULT,                 */
   1,   /* HB_P_NEGATE,               */
   1,   /* HB_P_NOOP,                 */
   1,   /* HB_P_NOT,                  */
   1,   /* HB_P_NOTEQUAL,             */
   1,   /* HB_P_OR,                   */
   4,   /* HB_P_PARAMETER,            */
   1,   /* HB_P_PLUS,                 */
   1,   /* HB_P_POP,                  */
   1,   /* HB_P_POPALIAS,             */
   3,   /* HB_P_POPALIASEDFIELD,      */
   2,   /* HB_P_POPALIASEDFIELDNEAR,  */
   3,   /* HB_P_POPALIASEDVAR,        */
   3,   /* HB_P_POPFIELD,             */
   3,   /* HB_P_POPLOCAL,             */
   2,   /* HB_P_POPLOCALNEAR,         */
   3,   /* HB_P_POPMEMVAR,            */
   3,   /* HB_P_POPSTATIC,            */
   3,   /* HB_P_POPVARIABLE,          */
   1,   /* HB_P_POWER,                */
   1,   /* HB_P_PUSHALIAS,            */
   3,   /* HB_P_PUSHALIASEDFIELD,     */
   2,   /* HB_P_PUSHALIASEDFIELDNEAR, */
   3,   /* HB_P_PUSHALIASEDVAR,       */
   0,   /* HB_P_PUSHBLOCK,            */
   0,   /* HB_P_PUSHBLOCKSHORT,       */
   3,   /* HB_P_PUSHFIELD,            */
   2,   /* HB_P_PUSHBYTE,             */
   3,   /* HB_P_PUSHINT,              */
   3,   /* HB_P_PUSHLOCAL,            */
   2,   /* HB_P_PUSHLOCALNEAR,        */
   3,   /* HB_P_PUSHLOCALREF,         */
   5,   /* HB_P_PUSHLONG,             */
   3,   /* HB_P_PUSHMEMVAR,           */
   3,   /* HB_P_PUSHMEMVARREF,        */
   1,   /* HB_P_PUSHNIL,              */
   11,  /* HB_P_PUSHDOUBLE,           */
   1,   /* HB_P_PUSHSELF,             */
   3,   /* HB_P_PUSHSTATIC,           */
   3,   /* HB_P_PUSHSTATICREF,        */
   0,   /* HB_P_PUSHSTR,              */
   0,   /* HB_P_PUSHSTRSHORT,         */
   3,   /* HB_P_PUSHSYM,              */
   2,   /* HB_P_PUSHSYMNEAR,          */
   3,   /* HB_P_PUSHVARIABLE,         */
   1,   /* HB_P_RETVALUE,             */
   3,   /* HB_P_SEND,                 */
   2,   /* HB_P_SENDSHORT,            */
   4,   /* HB_P_SEQBEGIN,             */
   4,   /* HB_P_SEQEND,               */
   1,   /* HB_P_SEQRECOVER,           */
   3,   /* HB_P_SFRAME,               */
   5,   /* HB_P_STATICS,              */
   0,   /* HB_P_STATICNAME,           */
   1,   /* HB_P_SWAPALIAS,            */
   1,   /* HB_P_TRUE,                 */
   1,   /* HB_P_ZERO,                 */
   1,   /* HB_P_ONE,                  */
   3,   /* HB_P_MACROFUNC,            */
   3,   /* HB_P_MACRODO,              */
   0,   /* HB_P_MPUSHSTR              */
   4,   /* HB_P_LOCALNEARADDINT,      */
   1,   /* HB_P_MACROPUSHREF          */
   9,   /* HB_P_PUSHLONGLONG          */
   3,   /* HB_P_ENUMSTART             */
   1,   /* HB_P_ENUMNEXT              */
   1,   /* HB_P_ENUMPREV              */
   1,   /* HB_P_ENUMEND               */
   3,   /* HB_P_SWITCH                */
   5,   /* HB_P_PUSHDATE,             */
   1,   /* HB_P_PLUSEQPOP,            */
   1,   /* HB_P_MINUSEQPOP,           */
   1,   /* HB_P_MULTEQPOP,            */
   1,   /* HB_P_DIVEQPOP,             */
   1,   /* HB_P_PLUSEQ,               */
   1,   /* HB_P_MINUSEQ,              */
   1,   /* HB_P_MULTEQ,               */
   1,   /* HB_P_DIVEQ,                */
   1,   /* HB_P_WITHOBJECTSTART,      */
   3,   /* HB_P_WITHOBJECTMESSAGE,    */
   1,   /* HB_P_WITHOBJECTEND,        */
   3,   /* HB_P_MACROSEND,            */
   1,   /* HB_P_PUSHOVARREF,          */
   1,   /* HB_P_ARRAYPUSHREF          */
   3,   /* HB_P_VFRAME                */
   4,   /* HB_P_LARGEFRAME            */
   4,   /* HB_P_LARGEVFRAME           */
   0,   /* HB_P_PUSHSTRHIDDEN         */
   5,   /* HB_P_LOCALADDINT           */
   1,   /* HB_P_MODEQPOP              */
   1,   /* HB_P_EXPEQPOP              */
   1,   /* HB_P_MODEQ                 */
   1,   /* HB_P_EXPEQ                 */
   1,   /* HB_P_DUPLUNREF             */
   0,   /* HB_P_MPUSHBLOCKLARGE,      */
   0,   /* HB_P_MPUSHSTRLARGE         */
   0,   /* HB_P_PUSHBLOCKLARGE,       */
   0,   /* HB_P_PUSHSTRLARGE          */
   2,   /* HB_P_SWAP                  */
   1,   /* HB_P_PUSHVPARAMS           */
   1,   /* HB_P_PUSHUNREF             */
   4,   /* HB_P_SEQALWAYS             */
   4,   /* HB_P_ALWAYSBEGIN           */
   1,   /* HB_P_ALWAYSEND             */
   1,   /* HB_P_DECEQPOP              */
   1,   /* HB_P_INCEQPOP              */
   1,   /* HB_P_DECEQ                 */
   1,   /* HB_P_INCEQ                 */
   3,   /* HB_P_LOCALDEC              */
   3,   /* HB_P_LOCALINC              */
   3,   /* HB_P_LOCALINCPUSH          */
   3,   /* HB_P_PUSHFUNCSYM           */
   3,   /* HB_P_HASHGEN               */
   1,   /* HB_P_SEQBLOCK              */
   0    /* HB_P_THREADSTATICS         */
};



ULONG pcode_len( BYTE* pPCode )
{
   ULONG   ulLen;

   if( *pPCode < HB_P_LAST_PCODE )
   {
      ulLen = ( ULONG ) s_bPCodeLen[ *pPCode ];

      if( ulLen == 0 )
      {
         switch( *pPCode )
         {
            case HB_P_PUSHBLOCK:
               return HB_PCODE_MKUSHORT( pPCode + 1 );
   
            case HB_P_PUSHBLOCKSHORT:
               return * ( pPCode + 1 );
   
            case HB_P_PUSHSTR:
               return 3 + HB_PCODE_MKUSHORT( pPCode + 1 );
   
            case HB_P_PUSHSTRSHORT:
               return 2 + * ( pPCode + 1 );
   
            case HB_P_PUSHSTRHIDDEN:
               return 4 + HB_PCODE_MKUSHORT( pPCode + 2 );
   
            case HB_P_PUSHBLOCKLARGE:
               return HB_PCODE_MKUINT24( pPCode + 1 );
   
            case HB_P_PUSHSTRLARGE:
               return 4 + HB_PCODE_MKUINT24( pPCode + 1 );
   
            case HB_P_THREADSTATICS:
               return 3 + ( ( ULONG ) HB_PCODE_MKUSHORT( pPCode + 1 ) << 1 );
   
            case HB_P_LOCALNAME:
            case HB_P_MODULENAME:
            case HB_P_STATICNAME:
            {
               BYTE*   pPCodeOld = pPCode;

               if( *pPCode == HB_P_STATICNAME )
                  pPCode++;

               pPCode += 3;
               while( *pPCode++ );
               return pPCode - pPCodeOld;
            }
         }
      }
      return ulLen;
   }
   printf( "Invalid PCode %d\n", *pPCode );
   return 0;
}


BOOL pcode_isjump( BYTE bPCode )
{
   return ( bPCode >= HB_P_JUMPNEAR && bPCode <= HB_P_JUMPTRUEFAR ) ||
          bPCode == HB_P_SEQBEGIN || bPCode == HB_P_SEQEND;
}


BOOL pcode_isjumpforce( BYTE bPCode )
{
   return bPCode == HB_P_JUMPNEAR ||
          bPCode == HB_P_JUMP ||
          bPCode == HB_P_JUMPFAR ||
          bPCode == HB_P_SEQEND;
}


LONG pcode_jumpoffset( BYTE* pPCode )
{
   switch( *pPCode )
   {
      case HB_P_JUMPNEAR:
      case HB_P_JUMPFALSENEAR:
      case HB_P_JUMPTRUENEAR:
         return * ( (signed char*) pPCode + 1 );

      case HB_P_JUMP:
      case HB_P_JUMPFALSE:
      case HB_P_JUMPTRUE:
         return HB_PCODE_MKSHORT( pPCode + 1 );

      case HB_P_JUMPFAR:
      case HB_P_JUMPFALSEFAR:
      case HB_P_JUMPTRUEFAR:
      case HB_P_SEQBEGIN:
      case HB_P_SEQEND:
         return HB_PCODE_MKINT24( pPCode + 1 );
   }
   return 0;
}


ULONG pcode_localno( BYTE* pPCode )
{
   switch( *pPCode )
   {
      case HB_P_POPLOCALNEAR:
      case HB_P_PUSHLOCALNEAR:
      case HB_P_LOCALNEARADDINT:
         return * ( (signed char*) pPCode + 1 );

      case HB_P_POPLOCAL:
      case HB_P_PUSHLOCAL:
      case HB_P_PUSHLOCALREF:
      case HB_P_LOCALDEC:
      case HB_P_LOCALINC:
      case HB_P_LOCALINCPUSH:
      case HB_P_LOCALADDINT:
         return HB_PCODE_MKSHORT( pPCode + 1 );
   }
   return 0;
}


BOOL pcode_isblock( BYTE bPCode )
{
   return bPCode == HB_P_PUSHBLOCK || bPCode == HB_P_PUSHBLOCKSHORT || bPCode == HB_P_PUSHBLOCKLARGE;
}


BOOL pcode_blockuseslocal( BYTE* pPCode, ULONG ulLocal )
{
   ULONG   ul, ulCount;

   if( * pPCode == HB_P_PUSHBLOCKSHORT )
      return 0;
   else if( * pPCode == HB_P_PUSHBLOCK )
      pPCode += 5;
   else
      pPCode += 6;

   ulCount = HB_PCODE_MKUSHORT( pPCode );

   for( ul = 0; ul < ulCount; ul++ )
   {
      if( ( ULONG ) HB_PCODE_MKUSHORT( pPCode + 2 + ul * 2 ) == ulLocal ) /* BCC warning: comparing sifgned and unsigned! Why? */
         return 1;
   }
   return 0;
}


int rtrace_set_optype( PCODECONTEXT* pCtx, ULONG ulPos )
{
   ULONG   ulLen, ul;

   while( 1 )
   {
      ulLen = pcode_len( pCtx->pPCode + ulPos );

      if( ! ulLen )
      {
         printf( "%s: PCode corrupted (loop)\n", pCtx->szFunc );
         return 0;
      }

      /* > but not >=, this let us process HB_P_ENDPROC, etc. */
      if( ulPos + ulLen > pCtx->ulPCodeLen )
      {
         printf( "%s: PCode corrupted (overflow)\n", pCtx->szFunc );
         return 0;
      }        

      if( ( pCtx->pOpType[ ulPos ] & ( OPTYPE_CODE | OPTYPE_PARAM ) ) == 0 )
      {
         pCtx->pOpType[ ulPos ] |= OPTYPE_CODE;

         for( ul = 1; ul < ulLen; ul++ )
         {
            if( pCtx->pOpType[ ulPos + ul ] & ( OPTYPE_CODE | OPTYPE_JUMPIN ) )
            {
               printf( "%s: PCode corrupted (overlay)\n", pCtx->szFunc );
               return 0;
            }
            pCtx->pOpType[ ulPos + ul ] |= OPTYPE_PARAM;
         }

         if( pcode_isjump( pCtx->pPCode[ ulPos ] ) )
         {
            ULONG  ulPos2 = ulPos + pcode_jumpoffset( pCtx->pPCode + ulPos );

            pCtx->pOpType[ ulPos ] |= OPTYPE_JUMP;

            if( ulPos2 >= pCtx->ulPCodeLen )
            {
               printf( "%s: PCode corrupted (overflow)\n", pCtx->szFunc );
               return 0;
            }        
            pCtx->pOpType[ ulPos2 ] |= OPTYPE_JUMPIN;

            if( pcode_isjumpforce( pCtx->pPCode[ ulPos ] ) )
            {
               ulPos = ulPos2;
               continue;
            }

            if( ! rtrace_set_optype( pCtx, ulPos2 ) )
               return 0;
         }
         else if( pCtx->pPCode[ ulPos ] == HB_P_SWITCH )
         {
            ULONG ul, ulCount = HB_PCODE_MKUSHORT( pCtx->pPCode + ulPos + 1 );

            ulPos += 3;
            for( ul = 0; ul < ulCount; ul++ )
            {
               if( ! rtrace_set_optype( pCtx, ulPos ) )
                  return 0;

               /* push instruction */
               ulPos += pcode_len( pCtx->pPCode + ulPos );
               if( ulPos > pCtx->ulPCodeLen )
               {
                  printf( "%s: PCode corrupted (overflow)\n", pCtx->szFunc );
                  return 0;
               }   
               /* jump instruction */
               if( pCtx->pPCode[ ulPos ] != HB_P_JUMPNEAR && 
                   pCtx->pPCode[ ulPos ] != HB_P_JUMP && 
                   pCtx->pPCode[ ulPos ] != HB_P_JUMPFAR )
               {
                  printf( "%s: PCode corrupted (invalid switch)\n", pCtx->szFunc );
                  return 0;
               }
               ulPos += pcode_len( pCtx->pPCode + ulPos );
               if( ulPos > pCtx->ulPCodeLen )
               {
                  printf( "%s: PCode corrupted (overflow)\n", pCtx->szFunc );
                  return 0;
               }        
            }
            continue;
         }
      }
      else if( pCtx->pOpType[ ulPos ] & OPTYPE_PARAM )
      {
         printf( "%s: PCode corrupted (overlay)\n", pCtx->szFunc );
         return 0;
      }

      if( pCtx->pPCode[ ulPos ] == HB_P_ENDPROC || pCtx->pPCode[ ulPos ] == HB_P_ENDBLOCK )
         break;

      ulPos += ulLen;
   }
   return 1;
}


int rtrace_assigned_unused( PCODECONTEXT* pCtx, ULONG ulPos, ULONG ulLocal )
{
   while( 1 )
   {
      if( pCtx->pOpType[ ulPos ] & 0x80 )
         return 1;

      pCtx->pOpType[ ulPos ] |= 0x80;

      if( pCtx->pPCode[ ulPos ] == HB_P_POPLOCAL || 
          pCtx->pPCode[ ulPos ] == HB_P_POPLOCALNEAR || 
          pCtx->pPCode[ ulPos ] == HB_P_PUSHLOCAL ||
          pCtx->pPCode[ ulPos ] == HB_P_PUSHLOCALNEAR ||
          pCtx->pPCode[ ulPos ] == HB_P_PUSHLOCALREF ||
          pCtx->pPCode[ ulPos ] == HB_P_LOCALINCPUSH )
      {
         if( pcode_localno( pCtx->pPCode + ulPos ) == ulLocal ) 
         {
            if( pCtx->pPCode[ ulPos ] == HB_P_PUSHLOCAL ||
                pCtx->pPCode[ ulPos ] == HB_P_PUSHLOCALNEAR ||
                pCtx->pPCode[ ulPos ] == HB_P_PUSHLOCALREF ||
                pCtx->pPCode[ ulPos ] == HB_P_LOCALINCPUSH )
               return 0;
            else
               return 1;

         }
      }

      if( pcode_isjump( pCtx->pPCode[ ulPos ] ) )
      {
         ULONG  ulPos2 = ulPos + pcode_jumpoffset( pCtx->pPCode + ulPos );

         if( pcode_isjumpforce( pCtx->pPCode[ ulPos ] ) )
         {
            ulPos = ulPos2;
            continue;
         }

         if( ! rtrace_assigned_unused( pCtx, ulPos2, ulLocal ) )
            return 0;
      }
      else if( pCtx->pPCode[ ulPos ] == HB_P_SWITCH )
      {
         ULONG ul, ulCount = HB_PCODE_MKUSHORT( pCtx->pPCode + ulPos + 1 );

         ulPos += 3;
         for( ul = 0; ul < ulCount; ul++ )
         {
            if( ! rtrace_assigned_unused( pCtx, ulPos, ulLocal ) )
               return 0;

            ulPos += pcode_len( pCtx->pPCode + ulPos );
            ulPos += pcode_len( pCtx->pPCode + ulPos );
         }
         continue;
      }

      if( pCtx->pPCode[ ulPos ] == HB_P_ENDPROC || pCtx->pPCode[ ulPos ] == HB_P_ENDBLOCK )
         break;

      ulPos += pcode_len( pCtx->pPCode + ulPos );
   }
   return 1;
}

void analyse_function( BYTE* pPCode, ULONG ulPCodeLen, char* szFunc )
{
   PCODECONTEXT*   pCtx;
   ULONG           ul, ul2, ul3;

   pCtx = ( PCODECONTEXT* ) hb_xgrab( sizeof( PCODECONTEXT ) );
   memset( pCtx, 0, sizeof( PCODECONTEXT ) );
   pCtx->pPCode = pPCode;
   pCtx->ulPCodeLen = ulPCodeLen;
   pCtx->pOpType = ( BYTE* ) hb_xgrab( ulPCodeLen );
   memset( pCtx->pOpType, 0, ulPCodeLen );
   pCtx->szFunc = szFunc; 

   if( pCtx->ulPCodeLen > 0 )
   {
      /* Calculate OpType and do general PCode recursive scan */
      if( rtrace_set_optype( pCtx, 0 ) )
      {
         for( ul = 0; ul < pCtx->ulPCodeLen; ul++ )
         {
            if( pCtx->pOpType[ ul ] == 0 )
            {
               printf( "%s: Dead PCode at %d\n", szFunc, ul );
               break;
            }
         }

/*
         for( ul = 0; ul < ulPCodeLen; ul++ )
            printf( "%02X ", pPCode[ ul ] );
         printf( "\n" );
         for( ul = 0; ul < ulPCodeLen; ul++ )
            printf( "%02X ", pCtx->pOpType[ ul ] );
         printf( "\n" );
*/
                                        
         /* Locals analysis */
         if( pPCode[ 0 ] == HB_P_FRAME || 
             pPCode[ 0 ] == HB_P_VFRAME )
         {
            pCtx->ulLocals = pPCode[ 1 ];
            pCtx->ulParams = pPCode[ 2 ];
         }
         else if( pPCode[ 0 ] == HB_P_LARGEFRAME || 
                  pPCode[ 0 ] == HB_P_LARGEVFRAME )
         {
            pCtx->ulLocals = HB_PCODE_MKUSHORT( pPCode + 1 );
            pCtx->ulParams = pPCode[ 3 ];
         }
/*
         Frames is absent in empty functions, static init functions
         else 
            printf( "%s: No frame found\n", szFunc );
*/

         if( pCtx->ulLocals > 0 )
         {
//            printf( "Params:%d Locals:%d\n", pCtx->ulParams, pCtx->ulLocals );

            for( ul2 = pCtx->ulParams + 1; ul2 <= pCtx->ulParams + pCtx->ulLocals; ul2++ )
            {
               BOOL   bUsed = 0;

               for( ul = 0; ul < ulPCodeLen; ul++ )
               {
                  if( ( pCtx->pOpType[ ul ] & OPTYPE_CODE ) && 
                      ( pPCode[ ul ] == HB_P_POPLOCAL || 
                        pPCode[ ul ] == HB_P_POPLOCALNEAR || 
                        pPCode[ ul ] == HB_P_PUSHLOCAL ||
                        pPCode[ ul ] == HB_P_PUSHLOCALNEAR ||
                        pPCode[ ul ] == HB_P_PUSHLOCALREF ||
                        pPCode[ ul ] == HB_P_LOCALNEARADDINT ||
                        pPCode[ ul ] == HB_P_LOCALADDINT ||
                        pPCode[ ul ] == HB_P_LOCALDEC ||
                        pPCode[ ul ] == HB_P_LOCALINC ||
                        pPCode[ ul ] == HB_P_LOCALINCPUSH ) )
                  {
                     if( pcode_localno( pPCode + ul ) == ul2 )
                     {
                        bUsed = 1;

                        if( pPCode[ ul ] == HB_P_POPLOCAL || 
                            pPCode[ ul ] == HB_P_POPLOCALNEAR )
                        {
                           for( ul3 = 0; ul3 < ulPCodeLen; ul3++ )
                              pCtx->pOpType[ ul3 ] &= 0x7F;

                           pCtx->pOpType[ ul ] |= 0x80;

                           if( rtrace_assigned_unused( pCtx, ul + pcode_len( pPCode + ul ), ul2 ) )
                           {
                              BOOL    bBlock = 0;
                              USHORT  usLine = 0;
   

                              for( ul3 = 0; ul3 < ulPCodeLen; ul3++ )
                              {
                                 if( ( pCtx->pOpType[ ul3 ] & OPTYPE_CODE ) )
                                 {
                                    if( pcode_isblock( pPCode[ ul3 ] ) &&
                                        pcode_blockuseslocal( pPCode + ul3, ul2 ) )
                                    {
                                       bBlock = 1;
                                       break;
                                    }
                                    if( ul3 < ul && pPCode[ ul3 ] == HB_P_LINE )
                                       usLine = HB_PCODE_MKUSHORT( pPCode + ul3 + 1 );
                                 }
                              }
                              if( ! bBlock )
                                 printf( "%s: Local %d assigned at offset %d (line %d) but unused\n", szFunc, ul2 - pCtx->ulParams, ul, usLine );
                           }
                        }
                     }
                  }
               }

               if( ! bUsed )
               {
                  BOOL  bBlock = 0;
                  
                  for( ul3 = 0; ul3 < ulPCodeLen; ul3++ )
                  {
                     if( ( pCtx->pOpType[ ul3 ] & OPTYPE_CODE ) &&
                         pcode_isblock( pPCode[ ul3 ] ) &&
                         pcode_blockuseslocal( pPCode + ul3, ul2 ) )
                     {
                        bBlock = 1;
                        break;
                     }
                  }
                  if( ! bBlock ) 
                     printf( "%s: Local %d declared but unused\n", szFunc, ul2 - pCtx->ulParams );
               }
            }
         }
      }
   }        
   else
      printf( "%s: PCode corrupted (no PCode found)\n", szFunc );

   hb_xfree( pCtx->pOpType );
   hb_xfree( pCtx );
}


/*
  ulong   signature;  0xC0 HRB
  ushort  version;
  ulong   symbolcount;
  symbols:
     char[]   name;
     uchar    0x00;
     uchar    scope;
     uchar    type;     
  ulong   functioncount;
  functions:
     char[]   name;
     uchar    0x00;
     ulong    size;
     uchar[]  pcode;
*/
void analyse_hrbbody( BYTE* pHrb, ULONG ulHrbLen, char* szFile )
{
   ULONG  ulSymbolCount, ulFunctionCount, ulIndex, ulOffset, ulLen;

   if( szFile )
      printf( "%s\n", szFile );

   if( ! pHrb )
      return;

   if( ulHrbLen < 10 || memcmp( pHrb, "\xC0HRB", 4 ) )
   {
      printf( "Invalid .hrb file 1\n" );
      return;
   }

   ulSymbolCount = HB_PCODE_MKLONG( pHrb + 6 );
/*   printf( "Symbol count: %d\n", ulSymbolCount );  */
   ulOffset = 10;
   for( ulIndex = 0; ulIndex < ulSymbolCount; ulIndex++ )
   {
/*      printf( "Symbol: %s\n", (char*) pHrb + ulOffset ); */
      while( ulOffset < ulHrbLen && pHrb[ ulOffset++ ] );
      ulOffset += 2;
      if( ulOffset > ulHrbLen )
      {
         printf( "Invalid .hrb file 2\n" );
         return;
      }
   }

   if( ulOffset + 4 > ulHrbLen )
   {
      printf( "Invalid .hrb file 3\n" );
      return;
   }

   ulFunctionCount = HB_PCODE_MKLONG( pHrb + ulOffset );
/*   printf( "Function count: %d\n", ulFunctionCount ); */
   ulOffset += 4;
   for( ulIndex = 0; ulIndex < ulFunctionCount; ulIndex++ )
   {
      char*  szFunc = ( char* ) pHrb + ulOffset;

      while( ulOffset < ulHrbLen && pHrb[ ulOffset++ ] );
      if( ulOffset + 4 > ulHrbLen )
      {
         printf( "Invalid .hrb file 4\n" );
         return;
      }
      ulLen = HB_PCODE_MKLONG( pHrb + ulOffset );
      ulOffset += 4;
      if( ulOffset + ulLen > ulHrbLen || ulLen == 0 )
      {
         printf( "Invalid .hrb file 5\n" );
         return;
      }
      analyse_function( pHrb + ulOffset, ulLen, szFunc );
      ulOffset += ulLen;
   }
}


HB_FUNC( ANALYSE_HRBBODY )
{
   analyse_hrbbody( ( BYTE* ) hb_parc( 1 ), hb_parclen( 1 ), hb_parc( 2 ) );
}

